


#include <Arduino.h>
//#include "DEBUG.h"
#include "nbiot.h"
#include <stdlib.h>
#include "parsecmd.h"
#include "difftime.h"

/**
 * Constructor for initializing nbiot object with default values.
 *
 * @param srl The serial stream for communication.
 *
 * @return None
 *
 * @throws None
 */
nbiot::nbiot(Stream &srl):parsecmd(srl){
    strcpy(apnName,apnNameX);
    strcpy(bandName,bandNameX);
    setupServer(mqttIpX,mqttUserNameX,mqttPassWordX,deviceIdX);
}

/**
 * Constructor for initializing nbiot object with APN, band, and serial stream.
 *
 * @param apn The Access Point Name to set.
 * @param band The NBIOT frequency band to set.
 * @param srl The serial stream for communication.
 *
 * @return None
 *
 * @throws None
 */
nbiot::nbiot(char *apn, char *band, Stream &srl):parsecmd(srl){
    strcpy(apnName,apn);
    strcpy(bandName,band);
    setupServer(mqttIpX,mqttUserNameX,mqttPassWordX,deviceIdX);
}

nbiot::~nbiot(){
}

/**
 * Sends the "AT" command to the nbiot device multiple times to wake it up.
 *
 * @param times the number of times to send the "AT" command.
 *
 * @return 0 if the nbiot device was not responding(not wakeing up), 1 if the nbiot device was responding.
 *
 * @throws None
 */
int nbiot::wakeUp(int times,uint32_t intervalTime){
    for(int i=0;i<times;i++){
        cSend("AT");
        if (i==0){
            if (getResponse("OK",200)==1) return 1;
        } else if (i==1){
            if (getResponse("OK")==1) return 1;
        } else {
            if (getResponse("OK")==1) return 1;
            delay(intervalTime);
        }
    }
    return 0;
}

int nbiot::setupNetwork(){
    if (wakeUp(3,200)==0) return 1;
    cSend("AT+QSCLK=0");
    if (getResponse("OK",1200)!=1) return 2;
    cSend("AT+QCGDEFCONT= \"IP\",\"%s\"",apnName);
    if (getResponse("OK",3600)!=1) return 3;
    cSend("AT+QBAND= %s",bandName);
    int i;
    for(i=0;i<10;i++)
        if (getResponse("OK",800)==1) break;
    if (i>=10) return 4;   
    //cSend("AT+IPR=115200");
    //if (getResponse("OK",500)!=1) return 5;
    //wakeUp(5,1000);
    cSend("AT+CFUN=1");
    if (getResponse("OK",6000)!=1) return 6;
    cSend("AT+QRST= 1");
    if (getResponse("OK",1200)!=1) return 7;
    return 0;
}

int nbiot::setupNetwork(char *apn,char *band){
    strcpy(apnName,apn);
    strcpy(bandName,band);
    return setupNetwork();
}

int nbiot::checkNetwork(){
    if (wakeUp(3)==0) return 1;

    cSend("AT+QSCLK=0");
    if (getResponse("OK",500)!=1) return 2; 
    
    cSend("AT+CPIN?");
    int errCode;
    if ((errCode=getResponse("+CPIN:","ERROR",2000))!=1){
        if (errCode==0) return 3;                                   // 檢查SIM卡狀態,未反應
        if (errCode==-1) return 4;                                  // SIM卡未插入                               
    };

    int i,j;
    for(i=0;i<36;i++){
        cSend("AT+CEREG?");
        if (getResponse("+CEREG:",1000)!=1) {  // 檢查NBIOT網路註冊狀態
            delay(1000);          
        }else{
            getParam(",");  
            if (1==atoi(nextParam())) break; 
            delay(1000);
        };
    };
    if (i>=36) return 5;

    cSend("AT+CSQ");
    for(i=0;i<10;i++){
        errCode=getResponse("+CSQ:","ERROR",800);
        if (errCode==1) break;
        if (errCode==0) continue;
        if (errCode==-1) return 7;    
    };
    if (i>=10) return 6;

    _rssi=atoi(getParam(","))*2-113;
     
    cSend("AT+QSCLK=1");
   
    return 0;                                                       // NBIOT網路正常
}

int nbiot::checkNetwork(int *rssi){
    int rcode;
    rcode = checkNetwork();
    *rssi = _rssi;
    return rcode;
}

int nbiot::getRSSI(int *errCode){           // return rssi. Return old rssi when getting RSSI operation is error.
    int _errCode;
    cSend("AT+CSQ");
    if ((_errCode=getResponse("+CSQ:","ERROR",800))!=1){
        if (errCode!=0) {
            if (_errCode==0) *errCode=1;
            if (_errCode==-1) *errCode=2;
        };
        return _rssi;    
    };
    if (errCode!=0) *errCode=0;
    _rssi=atoi(getParam(","))*2-113;
    return _rssi;    
}

/**
+ * Initializes the nbiot object with the provided MQTT server IP, username, password, and device ID.
+ *
+ * @param mqIp The IP address or domain name of the MQTT server.
+ * @param mqUserName The username for connecting to the MQTT server.
+ * @param mqPasswd The password for connecting to the MQTT server.
+ * @param dvID The device ID.
+ *
+ * @throws None
+ */
void nbiot::setupServer(char *mqIp, char *mqUserName, char *mqPasswd, char *dvID,int keepalive){
    strcpy(mqttIP,mqIp);
    strcpy(mqttUserName,mqUserName);
    strcpy(mqttPassWord,mqPasswd);
    strcpy(deviceID,dvID);
    mqttPort=mqttPortX;
    mID=1;
    keepAliveTime=keepalive;
}

/**
 * Sets up the NBIoT module with the given MQTT server details and device ID.
 *
 * @param mqPort the MQTT server port number
 * @param mqIp the MQTT server IP address or domain name
 * @param mqPort the MQTT server port number (duplicate parameter, should be removed)
 * @param mqUserName the MQTT server username
 * @param mqPasswd the MQTT server password
 * @param dvID the device ID
 * @param keepalive the MQTT keep alive time (default is 5 seconds)
 *
 * @throws None
 */
void nbiot::setupServer(uint16_t mqPort, char *mqIp, char *mqUserName, char *mqPasswd, char *dvID,int keepalive){
    strcpy(mqttIP,mqIp);
    strcpy(mqttUserName,mqUserName);
    strcpy(mqttPassWord,mqPasswd);
    strcpy(deviceID,dvID);
    mqttPort=mqPort;
    mID=0;
    keepAliveTime=keepalive;
}

int nbiot::start(){
    if (wakeUp(10)==0) return 1;

    char *ans;
    int i;
    
    cSend("AT+QSCLK= 0");
    if (getResponse("OK",500)!=1) return 2;

    for(i=0;i<36;i++){
        cSend("AT+CEREG?");
        if (getResponse("+CEREG:",500)!=1){ // 檢查NBIOT網路註冊狀態
            delay(1200);
        } else {
            getParam(",");  
            if (1==atoi(nextParam())) break;
            delay(1200);
        };
    };
    if (i>=36) return 3;
    
    cSend("AT+QMTCFG= \"KEEPALIVE\",%d,%d ",tID,keepAliveTime);
    if (getResponse("OK",2000)!=1) return 4;

    cSend("AT+QMTOPEN=%d,\"%s\",%d",tID,mqttIP,mqttPort);
    for(i=0;i<35;i++)
        if (getResponse("+QMTOPEN:",1000)==1) break;
    if (i>=35) return 5;
    ans=getParam(",");
    if (strcmp(ans,"0")!=0) return 7;
    ans=nextParam();
    if (strcmp(ans,"0")!=0) return 8;

    cSend("AT+QMTCONN= %d,\"%s\",\"%s\",\"%s\"",tID,deviceID,mqttUserName,mqttPassWord);
    for(i=0;i<10;i++)
        if (getResponse("OK",800)==1) break;
    if (i>=10) return 9;   
    
    for(i=0;i<10;i++)
        if (getResponse("+QMTCONN:",5000)==1) break;
    if (i>=10) return 10;
    ans=getParam(",");
    if (strcmp(ans,"0")!=0) return 11;
    ans=nextParam();
    if (strcmp(ans,"0")!=0) return 12;
    return 0;
}


/**
 * Disconnects from the MQTT broker and closes the TCP connection.
 *
 * @return 0 if successful, otherwise an error code:
 *         1 if the disconnect command failed
 *         2 if the disconnect response failed
 *         3 if the TCP ID does not match the target ID
 *         4 if the second parameter is not "0"
 *         5 if the close command failed
 *         6 if the close response failed
 *         7 if the TCP ID does not match the target ID
 *         8 if the second parameter is not "0"
 */
int nbiot::end(){
    char *ans;
    int tcpID;
    /*
    // Disconnect from the MQTT broker
    cSend("AT+QMTDISC= %d",tID);
    if (getResponse("OK",200)!=1) return 1;
    if (getResponse("+QMTDISC:",200)!=1) return 2;
    ans=getParam(",");
    tcpID=atoi(ans);
    if (tcpID!=tID) return 3;
    ans=nextParam();
    if (strcmp(ans,"0")!=0) return 4;
    */

    // Close the TCP connection
    cSend("AT+QMTCLOSE=%d",tID);
    if (getResponse("OK",200)!=1) return 5;
    if (getResponse("+QMTCLOSE:",200)!=1) return 6;
    if (atoi(getParam(","))!=tID) return 7;
    if (atoi(nextParam())!=0) return 8;

    cSend("AT+QSCLK= 1");
    //if (getResponse("OK",500)!=1) return 9;
    flush();
    return 0;
}

int nbiot::connected(){
    int i;
    send("AT+QMTCONN?");
    for(i=0;i<10;i++)
        if (getResponse("+QMTCONN:",500)==1) break;
    if (i>=10) return 0;
    if (tID!=atoi(getParam(","))) return 0;
    if (3!=atoi(nextParam())) return 0;
    return 1;           
}

/**
 * Publishes a message to a topic using the MQTT protocol.
 *
 * @param topic The topic to publish the message to.
 * @param payload The message payload to publish.
 *
 * @return 0 if the publication was successful, 0 < otherwise.
 *
 * @throws None
 */
int nbiot::pub(const char *topic, const char *payload,int mQos){
    send("AT+QMTPUB=%d,%d,%d,%d,\"%s\",%d,\"%s\"",tID,mID,mQos,mRetain,topic,strlen(payload),payload);
    int mid = mID;
    mID++;
    int i;

    for(i=0;i<3;i++){
        if (getResponse("OK",200)==1) break;        
    }
    if (i>=3) return 1;
    
    for(i=0;i<10;i++){
        if (getResponse("+QMTPUB:",1000)==1) break;
    }
    if (i>=10) return 2;
    
    char *ans = getParam(",");
    int tcpID=atoi(ans);
    if (tcpID!=tID) return 3;
    ans=nextParam();
    int mid_c=atoi(ans);
    if (mid_c!=mid) return 4;
    i=atoi(nextParam());
    if (i!=0 && i!=1) return 5;
    return 0;    
}


/**
 * Subscribes to a topic using the MQTT protocol.
 *
 * @param topic The topic to subscribe to.
 *
 * @return 0 if the subscription was successful, 0< otherwise.
 *
 * @throws None
 */
int nbiot::sub(const char *topic,int mQos){
    // Generate a unique message ID for the subscription
    int mid = mID;
    mID++;

    // Send the AT command to subscribe to the topic
    send("AT+QMTSUB=%d,%d,\"%s\",%d", tID, mid, topic, mQos);
        
    int i;
    // Check if the command was successful
    if (getResponse("OK",500)!=1) return 1;


    // Check if the response contains the expected parameters
    for(i=0;i<5;i++){
        if (getNextResponse("+QMTSUB:",1000)==1) break;
    }
    if (i>=5) return 2; 
    
    // Extract the TCP ID and message ID from the response
    char *ans=getParam(",");
    int tcpID=atoi(ans);
    if (tcpID!=tID) return 3;
    ans=nextParam();
    int mid_c=atoi(ans);
    if (mid_c!=mid) return 4;

    // Check if the subscription was successful
    i=atoi(nextParam());
    if (i!=0 && i!=1) return 5;

    // Subscription was successful
    return 0;
}

/**
 * Retrieves the subscription topic and message from the response of the AT command.
 *
 * @param t A pointer to a character array where the topic will be stored.
 * @param m A pointer to a character array where the message will be stored.
 *
 * @return 1 if the subscription topic and message are successfully retrieved, 0 otherwise.
 *
 * @throws None
 */
int nbiot::getSub(char *t, char *m,uint16_t maxResponeTime) {
    // Check if the response contains the expected start string
    listen(maxResponeTime);
    //if (getResponse("+QMTRECV:",maxResponeTime) != 1) return 0;
    if (getNextResponse("+QMTRECV:") != 1) return 0;
    

    // Extract the TCP ID from the response
    char *ans = getParam(",");
    int tcpID = atoi(ans);

    // Check if the extracted TCP ID matches the expected value
    if (tcpID != tID) return 0;
    
    // Extract the message ID from the response
    ans = nextParam();
    int mid = atoi(ans);

    // Extract the topic from the response
    ans = nextParam();
    strcpy(t, ans);

    // Extract the message from the response
    ans = nextParam();
    strcpy(m, ans);

    return 1;
}


/**
 * Retrieves the next subscription topic and message from the response of the AT command.
 *
 * @param t A pointer to a topic character array where the topic will be stored.
 * @param m A pointer to a overload character array where the overload will be stored.
 *
 * @return 1 if the next subscription topic and overload are successfully retrieved, 0 otherwise.
 *
 * @throws None
 */
int nbiot::getNextSub(char *t, char *m) {
    if (getNextResponse("+QMTRECV:") != 1) return 0;
    char *ans = getParam(",");
    int tcpID = atoi(ans);
    if (tcpID != tID) return 0;
    ans = nextParam();
    int mid = atoi(ans);
    ans = nextParam();
    strcpy(t, ans);
    ans = nextParam();
    strcpy(m, ans);
    return 1;
}
